'''
generate simiulated data for various model conditions
'''
from parameters import *
from functions import updated_signal
import numpy as np
import time
import os
from numba import jit

@jit(nopython=True)
def HS(S, s0, n, l):
	'''
	hill function
	'''
	return ( 1. + l*((S/s0)**n) )/( 1. + ((S/s0)**n) )

def set_ic(n):
	'''
	Initialize 2D arrays for all variables with randomized  initial conditions
	'''
	Ni = np.zeros((n, n))
	Di = np.zeros((n, n))
	Ji = np.zeros((n, n))
	Ii = np.zeros((n, n))
	Ri = np.zeros((n, n))
	Ti = np.zeros((n, n))

	for q in range(n):
		for w in range(n):
			Ni[q][w] = np.random.uniform(low=0, high=max(Nr, Ns))
			Di[q][w] = np.random.uniform(low=0, high=max(Dr, Ds))
			Ji[q][w] = np.random.uniform(low=0, high=max(Jr, Js))
			Ii[q][w] = np.random.uniform(low=0, high=max(Ir, Is))
			Ri[q][w] = np.random.uniform(low=0, high=max(Rr, Rs))
			Ti[q][w] = np.random.uniform(low=0, high=max(Tr, Ts))

	return Ni, Di, Ji, Ii, Ri, Ti

def integrate(N, D, J, I, R, T, tm, dt, Vt_value, Tt_value, hexagonal=False):
	'''
	integrate for time tm given initial conditions
	default geometry is square lattice
	'''

	Vt = Vt_value
	Tt = Tt_value

	for i in range(tm):

		Next = updated_signal(N, hexagonal=hexagonal)
		Dext = updated_signal(D, hexagonal=hexagonal)
		Jext = updated_signal(J, hexagonal=hexagonal)

		N = N + dt*( N0*HS(I,I0,p ,ln) - N*( kc*D + kt*Dext + kc*J + kt*Jext ) - g*N )
		D = D + dt*( D0*HS(I,I0,p ,ld)*HS(kt*R*Vt/gI,V0,p,ldr) - D*( kc*N + kt*Next ) - g*D )
		J = J + dt*( J0*HS(I,I0,pj,lj)*HS(kt*T*Tt/gI,TJ,nT,lT) - J*( kc*N + kt*Next ) - g*J )
		I = I + dt*( kt*N*(Dext+Jext) - gI*I )
		R = R + dt*( R0*HS(I,I0,p ,lr) - R*kt*Vt - g*R )
		T = T + dt*( T0 - kt*T*Tt - g*T )

	return N, D, J, I, R, T


def generate_data():
	np.random.seed(100)
	n = 10 						# lattice size
	hexagonal = True  			# use hexagonal lattice
	Vt_vec = np.array([0., 10., 50., 100., 150., 200., 220, 240, 260, 280, 300., 500., 1000., 2000., 3000., 5000.,
					   10000.]) # values of external VEGF to simulate, each value generate a data file
	Tt = 0. 					# external TNF signal, not used in this paper
	nsim = 3  					# 50
	T1, T2 = 100, 500  			# time for initial relaxation and simulation afterwards
	dt = 0.1  					# timestep

	for i in range(Vt_vec.size):
		for q in range(nsim):
			print(Vt_vec[i], q + 1)
			# first quick simulation starting from random
			Nrnd, Drnd, Jrnd, Irnd, Rrnd, Trnd = set_ic(n)
			Ni, Di, Ji, Ii, Ri, Ti = integrate(Nrnd, Drnd, Jrnd, Irnd, Rrnd, Trnd, int(T1 / dt), dt, 0., 0.,
											   hexagonal=hexagonal)

			N, D, J, I, R, T = integrate(Ni, Di, Ji, Ii, Ri, Ti, int(T2 / dt), dt, Vt_vec[i], Tt, hexagonal=hexagonal)
			np.savetxt('sim_data/delta_special_' + str(int(Vt_vec[i])) + '_' + str(int(Tt)) + '.txt', D)
			np.savetxt('sim_data/notch_special_' + str(int(Vt_vec[i])) + '_' + str(int(Tt)) + '.txt', N)

def main():
	if not os.path.isdir('sim_data/'):
		os.mkdir('sim_data/')
		generate_data()

if __name__=='__main__':
	main()

